Imports System
Imports System.Collections

' This is a sample implementation of a proxy pattern.
' We are wrapping System.Collections.IDictionary objects and
' System.ICloneable objects with a single class.  The wrapped
' object must implement ICloneable.  This class 
' provides a "cloning on write" behavior to allow
' multithreaded use of seperate dictionaries. 
Public Class LazyCloneDictionary
    Implements IDictionary
    Implements ICloneable

    ' The Dictionary object that this object is a proxy for.
    Private dict As IDictionary

    ' This is the number of proxy objects that share the same
    ' underlying dictionary.
    Private refCount As MutableInteger

    ' Constructor
    ' d - IDictionary object this object should be a proxy for.
    Public Sub New(ByVal d As IDictionary)
        ' Clone wrapped dictionary if 
        dict = CType(CType(d, ICloneable).Clone(), IDictionary)
        refCount = New MutableInteger(1)
    End Sub 'New

    ' Creates a copy of the dictionary
    Public Function Clone() As Object _
                    Implements ICloneable.Clone
        Dim theClone As LazyCloneDictionary = _
          CType(MemberwiseClone(), LazyCloneDictionary)
        SyncLock refCount
            refCount.Value = refCount.Value + 1
        End SyncLock
        Return theClone
    End Function 'Clone

    Private Sub ensureUnderlyingDictionaryNotShared()
        SyncLock refCount
            If refCount.Value > 1 Then
                ' this will copy the entries
                dict = CType(CType(dict, ICloneable).Clone(), _
                             IDictionary)
                ' update shared refCount
                refCount.Value = refCount.Value - 1
                ' create new local refCount
                refCount = New MutableInteger(1)
            End If
        End SyncLock
    End Sub 'ensureUnderlyingDictionaryNotShared

    ' number of key-value pairs.
    Public ReadOnly Property Count() As Integer _
                             Implements ICollection.Count
        Get
            Return dict.Count
        End Get
    End Property

    ' indicates if dictionary access is thread-safe
    Public ReadOnly Property IsSynchronized() As Boolean _
                          Implements IDictionary.IsSynchronized
        Get
            Return dict.IsSynchronized
        End Get
    End Property

    ' object used for synchronization
    Public ReadOnly Property SyncRoot() As Object _
                            Implements IDictionary.SyncRoot
        Get
            Return dict.SyncRoot
        End Get
    End Property

    ' Copies the dictionary to a 1-dimensional System.Array,
    ' starting at the speficied index
    Public Sub CopyTo(ByVal a As Array, ByVal i As Integer) _
               Implements ICollection.CopyTo
        dict.CopyTo(a, i)
    End Sub 'CopyTo

    Function IEnumGetEnumerator() As IEnumerator _
            Implements IEnumerable.GetEnumerator
        Return dict.GetEnumerator()
    End Function 'IEnumerable.GetEnumerator

    ' indicates if fixed size   
    Public ReadOnly Property IsFixedSize() As Boolean _
                             Implements IDictionary.IsFixedSize
        Get
            Return dict.IsFixedSize
        End Get
    End Property

    ' indicates if read-only
    Public ReadOnly Property IsReadOnly() As Boolean _
                             Implements IDictionary.IsReadOnly
        Get
            Return dict.IsReadOnly
        End Get
    End Property

    ' keys in dictionary   
    Public ReadOnly Property Keys() As ICollection _
                             Implements IDictionary.Keys
        Get
            Return dict.Keys
        End Get
    End Property

    ' values in dictionary
    Public ReadOnly Property Values() As ICollection _
                             Implements IDictionary.Values
        Get
            Return dict.Values
        End Get
    End Property

    ' Determines whether the dictionary contains a specific key
    Public Function Contains(ByVal key As Object) As Boolean _
                    Implements IDictionary.Contains
        Return dict.Contains(key)
    End Function 'Contains

    Function GetEnumerator() As IDictionaryEnumerator _
            Implements IDictionary.GetEnumerator
        Return dict.GetEnumerator()
    End Function 'IDictionary.GetEnumerator

    ' gets or sets value associated with key   
    Default Public Property Item(ByVal index As Object) As Object _
                  Implements IDictionary.Item
        Get
            Return dict(index)
        End Get
        Set(ByVal value As Object)
            ensureUnderlyingDictionaryNotShared()
            dict(index) = value
        End Set
    End Property

    ' Add to dictionary
    Public Sub Add(ByVal key As Object, ByVal value As Object) _
               Implements IDictionary.Add
        ensureUnderlyingDictionaryNotShared()
        dict.Add(key, value)
    End Sub 'Add

    ' Clear dictionary
    Public Sub Clear() Implements IDictionary.Clear
        ensureUnderlyingDictionaryNotShared()
        dict.Clear()
    End Sub 'Clear

    ' Removes the element with the specified key from the dictionary.
    Public Sub Remove(ByVal key As Object) _
               Implements IDictionary.Remove
        ensureUnderlyingDictionaryNotShared()
        dict.Remove(key)
    End Sub 'Remove

    ' Determines whether two Object instances are equal.
    Public Overloads Overrides Function Equals(ByVal o As Object) _
                                        As Boolean
        Return CType(dict, Object).Equals(o)
    End Function 'Equals

    ' Hash function for current type
    Public Overrides Function GetHashCode() As Integer
        Return CType(dict, Object).GetHashCode()
    End Function 'GetHashCode
End Class 'LazyCloneDictionary

' Reference counting class
Public Class MutableInteger
    Private val As Integer

    ' reference tracking integer
    Public Sub New(ByVal value As Integer)
        val = value
    End Sub 'New

    ' gets or sets the reference count   
    Public Property Value() As Integer
        Get
            Return val
        End Get
        Set(ByVal value As Integer)
            val = value
        End Set
    End Property
End Class 'MutableInteger
